<?php

namespace App\Services\Reports;

use App\Models\Feedback;
use App\Models\Masters\BlockMaster;
use App\Models\Masters\CityMaster;
use App\Models\Masters\ProgramMaster;
use App\Models\Masters\SchemeMaster;
use App\Models\Transactions\BeneficiaryImport;
use App\Services\DashboardService;
use App\Utilities\AppClass;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use DB;
use Auth;
use Carbon\Carbon;

class PresentationReportService
{

    private $dashboardService;
    private $presentationEntities;

    public function __construct(DashboardService $dashboardService, PresentationEntities $presentationEntities)
    {
        $this->dashboardService = $dashboardService;
        $this->presentationEntities = $presentationEntities;
    }

    public function getEscalationDashboard(Request $request)
    {
        $data = new AppClass();
        $data->view = 'reports.presentation';
        $data->blocks = BlockMaster::whereStatus(BlockMaster::STATUS_ACTIVE)->get();
        $data->scheme = SchemeMaster::whereStatus(SchemeMaster::STATUS_ACTIVE)->get();
        $data->program = ProgramMaster::whereStatus(ProgramMaster::STATUS_ACTIVE)->get();
        $data->district = CityMaster::whereStatus(CityMaster::STATUS_ACTIVE)->get();
        return $data;
    }

    public function getCallsDetails($data, $args)
    {
        $sql = $this->getCoreSqlFeedBack($args);

        $benSql = $this->getCoreSqlBeneficiary($args);
        $benSql->leftJoin('call_skip_reason', 'call_skip_reason.id', 'beneficiary_import.skip_reason');
        $benSql->groupBy('beneficiary_import.skip_reason');
        $benSql->select(DB::raw('COUNT(beneficiary_import.id) as count'), 'call_skip_reason.is_answered', 'call_skip_reason.name');
        $calls = $benSql->get();

        foreach ($data as $details) {
            if ($details->title == PresentationEntities::TOTAL_DISTRICTS_COVERED) {
                $disSql = clone $sql;
                $disSql->groupBy('beneficiary_import.district_id');
                $count = $disSql->get()->count();
                $details->count = isset($count) ? $count : 0;
            } else if ($details->title == PresentationEntities::TOTAL_FEEDBACK_RECEIVED) {
                $feedSql = clone $sql;
                $feedSql->where('feedback.is_compliant', 0);
                $feedSql->where('feedback.status', Feedback::STATUS_ACTIVE);
                $feedSql->whereNull('beneficiary_import.skip_reason');
                $count = $feedSql->count();
                $details->count = isset($count) ? $count : 0;
            } else if ($details->title == PresentationEntities::TOTAL_COMPLAINTS_RECEIVED) {
                $comSql = clone $sql;
                $comSql->where('feedback.is_compliant', Feedback::IS_COMPLIANT);
                $comSql->where('feedback.status', Feedback::STATUS_ACTIVE);
                $comSql->whereNull('beneficiary_import.skip_reason');
                $count = $comSql->count();
                $details->count = isset($count) ? $count : 0;
            } else if ($details->title == PresentationEntities::ANSWERED_CALLS) {
                foreach ($calls as $callRow) {
                    if ($callRow->is_answered == 1) {
                        $details->count += $callRow->count;
                    }
                }
            } else if ($details->title == PresentationEntities::UN_ANSWERED_CALLS) {
                foreach ($calls as $callRow) {
                    if (!is_null($callRow->is_answered) && $callRow->is_answered == 0) {
                        $details->count += $callRow->count;
                    }
                }
            }
        }

        $data = $this->getAnsweredCounts($data);
        return $data;
    }

    public function getAnsweredCounts($data)
    {
        $answered = 0;
        $unanswered = 0;
        $existingAnswered = 0;
        foreach ($data as $row) {
            if ($row->title == PresentationEntities::TOTAL_FEEDBACK_RECEIVED || $row->title == PresentationEntities::TOTAL_COMPLAINTS_RECEIVED) {
                $answered += $row->count;
            } else if ($row->title == PresentationEntities::UN_ANSWERED_CALLS) {
                $unanswered = $row->count;
            } else if ($row->title == PresentationEntities::ANSWERED_CALLS) {
                $existingAnswered = $row->count;
            }
        }

        $totalCalls = 0;
        foreach ($data as $row) {
            if ($row->title == PresentationEntities::ANSWERED_CALLS) {
                $row->count = $answered + $row->count;
            } else if ($row->title == PresentationEntities::TOTAL_CALLS_MADE) {
                $row->count = $answered + $unanswered + $existingAnswered;
                $totalCalls = $row->count;
            }
        }

        if ($totalCalls > 0) {
            foreach ($data as $row) {
                if (isset($row->showPercent) && $row->showPercent) {
                    $row->percentage = calcPercentage($row->count, $totalCalls);
                }
            }
        }

        return $data;
    }

    public function answeredComparisionChart(Request $request)
    {
        $args = $this->dashboardService->processFilters($request);

        $dates = new Collection();
        $currentMonth = new AppClass();
        $previousMonth = new AppClass();

        $previousMonth->fromDate = Carbon::parse($args->fromDate)->subMonths(1)->format('Y-m-01');
        $previousMonth->toDate = Carbon::parse($previousMonth->fromDate)->format('Y-m-t');
        $dates->add($previousMonth);

        $currentMonth->fromDate = Carbon::parse($args->fromDate)->format('Y-m-01');
        $currentMonth->toDate = Carbon::parse($args->toDate)->format('Y-m-t');
        $dates->add($currentMonth);

        $data = new AppClass();
        $data->title = 'Answered & Unanswered comparision last 2 months';
        $data->renderId = 'answered-comparision-chart';
        $data->yaxisTitle = 'calls';
        $data->series = new Collection();
        $data->xaxisColumns = new Collection();

        $monthData = new AppClass();
        $monthData->answered = new Collection();
        $monthData->unanswered = new Collection();

        foreach ($dates as $dateObj) {
            $newArgs = clone $args;
            $newArgs->fromDate = $dateObj->fromDate;
            $newArgs->toDate = $dateObj->toDate;
            $monthName = Carbon::parse($newArgs->fromDate)->format('F');
            $data->xaxisColumns->add($monthName);
            $calls = $this->presentationEntities->totalCallsEntities();
            $calls = $this->getCallsDetails($calls, $newArgs);

            foreach ($calls as $details) {
                if ($details->title == PresentationEntities::ANSWERED_CALLS) {
                    $total = new AppClass();
                    $total->name = 'Answered';
                    $total->count = $details->count;
                    $total->month = $monthName;
                    $monthData->answered->add($total);
                } else if ($details->title == PresentationEntities::UN_ANSWERED_CALLS) {
                    $total = new AppClass();
                    $total->name = 'Unanswered';
                    $total->count = $details->count;
                    $total->month = $monthName;
                    $monthData->unanswered->add($total);
                }
            }
        }

        $total = new AppClass();
        $total->name = 'Answered';
        $total->data = $monthData->answered->pluck('count');
        $data->series->add($total);

        $total = new AppClass();
        $total->name = 'Un Answered';
        $total->data = $monthData->unanswered->pluck('count');
        $data->series->add($total);

        return response()->json($data);
    }

    public function getTotalCallsSummary(Request $request)
    {
        $args = $this->dashboardService->processFilters($request);
        $data = $this->presentationEntities->totalCallsEntities();
        $data = $this->getCallsDetails($data, $args);
        $args->nextRenderId = 'programwise-calls';
        $args->title = 'Total Calls Summary';
        $html = view('reports.presentation-report', compact('data', 'args'))->render();
        $response = [
            'status' => true,
            'redirect' => null,
            'renderId' => '#dashboard-content',
            'html' => $html
        ];
        return response()->json($response);
    }

    public function getIcons($index)
    {
        $icons = [
            'fa fa-window-close',
            'fa fa-bell-slash',
            'fa fa-user',
            'fa fa-check',
            'fa fa-files-o',
            'fa fa-mobile',
            'fa fa-exclamation-triangle',
        ];
        return $icons[$index];
    }

    public function getDesignClass($index)
    {
        $class = [
            'text-danger',
            'text-success',
            'text-warning',
            'text-info',
            'text-primary',
        ];
        return $class[$index];
    }

    public function programwiseCallsEntities($args)
    {
        $data = new Collection();

        $sql = $this->getCoreSqlFeedBack($args);
        $sql->groupBy('beneficiary_import.program_id');
        $sql->select(DB::raw('COUNT(feedback.id) as count'), 'program_master.name');
        $programCounts = $sql->get();

        if ($programCounts) {
            foreach ($programCounts as $key => $program) {
                $districts = new AppClass();
                $districts->count = $program->count;
                $districts->title = $program->name;
                $districts->class = $this->getDesignClass($key) . ' ' . $this->getIcons($key);
                $districts->showLink = false;
                $data->add($districts);
            }
        }

        return $data;
    }

    public function getIndividualPrograms($args)
    {
        $data = new Collection();
        $sql = $this->getCoreSqlBeneficiary($args);
        $cloneSql = clone $sql;

        $sql->groupBy('beneficiary_import.program_id');
        $sql->select(
            'program_master.name',
            'beneficiary_import.program_id',
            DB::raw('COUNT(CASE WHEN feedback.id IS NULL THEN beneficiary_import.id END) as totalCalls'),
            DB::raw('COUNT(CASE WHEN beneficiary_import.skip_reason = ' . BeneficiaryImport::SKIP_REASON_WRONG_NUMBER . ' THEN beneficiary_import.id END) as wrongNumber'),
            DB::raw('COUNT(CASE WHEN beneficiary_import.skip_reason = ' . BeneficiaryImport::SKIP_REASON_NOT_ATTEND . ' THEN beneficiary_import.id END) as notAttend'),
        );

        $programCounts = $sql->get();

        if ($programCounts) {

            foreach ($programCounts as $key => $program) {

                $feSql = clone $cloneSql;
                $feSql->where('beneficiary_import.program_id', $program->program_id);
                $feSql->select(
                    DB::raw('COUNT(CASE WHEN feedback.is_compliant = 1 THEN feedback.id END) as complaintCount'),
                    DB::raw('COUNT(CASE WHEN feedback.is_compliant = 0 THEN feedback.id END) as feedbackCount'),
                );
                if (!is_null(Auth::user()->role)) {
                    $feSql->where('feedback.is_show', 1);
                }
                $getComplaints = $feSql->first();

                $feedback = $getComplaints->feedbackCount ? $getComplaints->feedbackCount : 0;
                $complaint = $getComplaints->complaintCount ? $getComplaints->complaintCount : 0;

                $node = new AppClass();
                $node->title = $program->name;
                $node->rows = new Collection();

                $totalCalls = $program->totalCalls + $feedback + $complaint;
                $program->title = 'TOTAL CALLS MADE';
                $program->count = $totalCalls;
                $items = $this->makeProgramEntities($program, 0);
                $node->rows->add($items);

                $program->title = 'TOTAL FEEDBACK RECEIVED';
                $program->count = $feedback;
                $items = $this->makeProgramEntities($program, 1);
                $items->showPercent = true;
                $items->percentage = calcPercentage($feedback, $totalCalls);
                $node->rows->add($items);

                $program->title = 'TOTAL COMPLAINTS RECEIVED';
                $program->count = $complaint;
                $items = $this->makeProgramEntities($program, 2);
                $items->showPercent = true;
                $items->percentage = calcPercentage($complaint, $totalCalls);
                $node->rows->add($items);

                $program->title = 'WRONG NUMBERS';
                $program->count = $program->wrongNumber;
                $items = $this->makeProgramEntities($program, 3);
                $items->showPercent = true;
                $items->percentage = calcPercentage($program->wrongNumber, $totalCalls);
                $node->rows->add($items);

                $program->title = 'NOT ANSWERED';
                $program->count = $program->notAttend;
                $items = $this->makeProgramEntities($program, 4);
                $items->showPercent = true;
                $items->percentage = calcPercentage($program->notAttend, $totalCalls);
                $node->rows->add($items);

                $data->add($node);
            }
        }

        return $data;
    }

    public function makeProgramEntities($program, $key)
    {
        $total = new AppClass();
        $total->count = $program->count;
        $total->title = $program->title;
        $total->class = $this->getDesignClass($key) . ' ' . $this->getIcons($key);
        $total->showLink = false;
        return $total;
    }

    public function gethighComplaintsProgram($args)
    {
        $data = new Collection();

        $programCounts = $this->getHighestComplaintsProgram($args);

        if ($programCounts) {

            $args->programId = $programCounts->program_id;
            $districtCounts = $this->getHighestComplaintsDistrict($args);

            $data = $this->presentationEntities->highComplaintsProgramEntities();
            foreach ($data as $program) {
                if ($program->title == PresentationEntities::PROGRAM_WITH_HIGH_COMPLAINTS) {
                    $program->subTitle = $programCounts->programName;
                    $program->count = '';
                } else if ($program->title == PresentationEntities::TOTAL_NO_OF_COMPLAINTS) {
                    $program->count = $programCounts->complaintsCount;
                } else if ($program->title == PresentationEntities::DISTRICT_WITH_HIGH_COMPLAINTS) {
                    $program->count = isset($districtCounts->complaintsCount) ? $districtCounts->complaintsCount : '';
                    $program->subTitle = isset($districtCounts->cityName) ? $districtCounts->cityName : '';
                }
            }
        }

        return $data;
    }

    public function highComplaintsProgramwise(Request $request)
    {
        $args = $this->dashboardService->processFilters($request);
        $data = $this->gethighComplaintsProgram($args);
        $args->title = 'High Complaints Data – Programwise';
        $html = view('reports.presentation-report', compact('data', 'args'))->render();
        $response = [
            'status' => true,
            'redirect' => null,
            'renderId' => '#high-complaints-program-wise',
            'html' => $html
        ];
        return response()->json($response);
    }

    public function getHighestComplaintsProgram($args)
    {
        $sql = $this->getCoreSqlFeedBack($args);
        $sql->groupBy('beneficiary_import.program_id');
        $sql->select(
            'program_master.name as programName',
            'beneficiary_import.program_id',
            DB::raw('COUNT(CASE WHEN feedback.is_compliant = 1 THEN feedback.id END) as complaintsCount')
        );
        $sql->orderBy('complaintsCount', 'desc');
        $programCounts = $sql->first();
        return $programCounts;
    }

    public function getHighestComplaintsDistrict($args)
    {
        $sql = $this->getCoreSqlFeedBack($args);
        $sql->groupBy('beneficiary_import.district_id');
        $sql->select(
            'city_master.name as cityName',
            'city_master.id as district_id',
            DB::raw('COUNT(CASE WHEN feedback.is_compliant = 1 THEN feedback.id END) as complaintsCount')
        );
        $sql->orderBy('complaintsCount', 'desc');
        $districtCounts = $sql->first();
        return $districtCounts;
    }

    public function gethighComplaintsDistrictwise($args)
    {
        $data = new Collection();
        $programCounts = $this->getHighestComplaintsProgram($args);
        $args->districtName = '';
        if ($programCounts) {

            $districtData = $this->getHighestComplaintsDistrict($args);

            if ($districtData) {
                $args->districtName = $districtData->cityName;
                $sql = $this->getCoreSqlFeedBack($args);
                $cloneSql = clone $sql;
                $sql->groupBy('beneficiary_import.program_id');
                $sql->where('beneficiary_import.district_id', $districtData->district_id);
                $sql->select(
                    'program_master.name as programName',
                    DB::raw('COUNT(CASE WHEN feedback.is_compliant = 1 THEN feedback.id END) as complaintsCount')
                );
                if (!is_null(Auth::user()->role)) {
                    $sql->where('feedback.is_show', 1);
                }
                $hightComp = $sql->get();

                if ($hightComp) {
                    foreach ($hightComp as $key => $program) {
                        $districts = new AppClass();
                        $districts->count = $program->complaintsCount;
                        $districts->title = $program->programName;
                        $districts->class = $this->getDesignClass($key) . ' ' . $this->getIcons($key);
                        $districts->showLink = false;
                        $data->add($districts);
                    }
                }

                $districts = new AppClass();
                $districts->count = $districtData->complaintsCount;
                $districts->title = 'No Of Complaints';
                $districts->class = $this->getDesignClass(1) . ' ' . $this->getIcons(1);
                $districts->showLink = false;
                $data->add($districts);

                $cloneSql->where('beneficiary_import.district_id', $districtData->district_id);
                $cloneSql->groupBy('beneficiary_import.block_id');
                $cloneSql->select(
                    'block_master.name as blockName',
                    DB::raw('COUNT(CASE WHEN feedback.is_compliant = 1 THEN feedback.id END) as complaintsCount')
                );
                $cloneSql->orderBy('complaintsCount', 'desc');
                $blockData = $cloneSql->first();

                $districts = new AppClass();
                $districts->count = '';
                $districts->subTitle = $blockData->blockName;
                $districts->title = 'Block with High Complaints';
                $districts->class = $this->getDesignClass(2) . ' ' . $this->getIcons(2);
                $districts->showLink = false;
                $data->add($districts);
            }
        }

        return $data;
    }

    public function highComplaintsDistrictwiseData(Request $request)
    {
        $args = $this->dashboardService->processFilters($request);
        $data = $this->gethighComplaintsDistrictwise($args);
        $args->title = 'High Complaints Data District wise - ' . $args->districtName . '';
        $html = view('reports.presentation-report', compact('data', 'args'))->render();
        $response = [
            'status' => true,
            'redirect' => null,
            'renderId' => '#high-complaints-program-wise',
            'html' => $html
        ];
        return response()->json($response);
    }

    public function getCategorywiseComplaints($args)
    {
        $data = new Collection();

        $sql = $this->getCoreSqlFeedBack($args);
        $sql->groupBy('beneficiary_import.program_id');
        $sql->select(
            'program_master.name',
            DB::raw('COUNT(CASE WHEN feedback.is_compliant = 1 THEN feedback.id END) as count')
        );
        $sql->orderBy('count', 'desc');
        $programCounts = $sql->get();

        if ($programCounts) {
            foreach ($programCounts as $key => $program) {
                $districts = new AppClass();
                $districts->count = $program->count;
                $districts->title = $program->name;
                $districts->class = $this->getDesignClass($key) . ' ' . $this->getIcons($key);
                $districts->showLink = false;
                $data->add($districts);
            }
        }
        return $data;
    }

    public function categorywiseComplaints(Request $request)
    {
        $args = $this->dashboardService->processFilters($request);
        $data = $this->getCategorywiseComplaints($args);
        $args->title = 'Category wise Complaints';
        $html = view('reports.presentation-report', compact('data', 'args'))->render();
        $response = [
            'status' => true,
            'redirect' => null,
            'renderId' => '#categorywise-complaints',
            'html' => $html
        ];
        return response()->json($response);
    }

    public function getSeparateProgramwiseCalls(Request $request)
    {
        $args = $this->dashboardService->processFilters($request);
        $programdata = $this->getIndividualPrograms($args);
        $html = '';

        foreach ($programdata as $key => $dataRow) {
            $args->title = $dataRow->title;
            $data = $dataRow->rows;
            $html .= view('reports.presentation-report', compact('data', 'args'))->render();
        }

        $response = [
            'status' => true,
            'redirect' => null,
            'renderId' => '#separte-programwise-calls',
            'html' => $html
        ];
        return response()->json($response);
    }

    public function getProgramwiseCalls(Request $request)
    {
        $args = $this->dashboardService->processFilters($request);
        $data = $this->programwiseCallsEntities($args);
        $args->title = 'Program wise Calls Made';
        $html = view('reports.presentation-report', compact('data', 'args'))->render();
        $response = [
            'status' => true,
            'redirect' => null,
            'renderId' => '#programwise-calls',
            'html' => $html
        ];
        return response()->json($response);
    }

    public function getDistrictwiseChart(Request $request)
    {
        $args = $this->dashboardService->processFilters($request);
        $sql = $this->getCoreSqlBeneficiary($args);
        $sql->groupBy('beneficiary_import.district_id');
        $sql->select(
            'city_master.name',
            DB::raw('COUNT(beneficiary_import.id) as totalCalls')
        );
        $sql->orderBy('totalCalls', 'desc');
        $districtCounts = $sql->get();

        $data = new AppClass();
        $data->title = 'District wise Calls Made';
        $data->renderId = 'districtwise-chart';
        $data->yaxisTitle = 'Calls';

        if ($districtCounts) {

            $data->xaxisColumns = $districtCounts->pluck('name');
            $data->series = new AppClass();
            $data->series->name = 'Calls';
            $data->series->data = new Collection();

            foreach ($districtCounts as $district) {
                $total = [$district->name, $district->totalCalls];
                $data->series->data->add($total);
            }
        }

        return response()->json($data);
    }

    public function getMaximumFeedbackChartPercentage(Request $request)
    {
        $feedbackCounts = $this->commmonFeedbackCharts($request);
        $data = new AppClass();
        $data->title = 'Maximum feedback received districts (Percentage)';
        $data->renderId = 'maximum-feedback-chart-percentage';
        $data->yaxisTitle = 'Percentage';

        if ($feedbackCounts) {

            foreach ($feedbackCounts as $feedback) {
                $feedback->feedbackPercentage = calcPercentage($feedback->feedbackCount, $feedback->totalCalls);
            }

            $sorted = $feedbackCounts->sortBy([
                ['feedbackPercentage', 'desc'],
            ])->values();

            $data->xaxisColumns = $sorted->pluck('name');
            $data->series = new Collection();

            $feedback = new AppClass();
            $feedback->name = 'Feedbacks Percentage';
            $feedback->data = $sorted->pluck('feedbackPercentage');
            $data->series->add($feedback);
        }

        return response()->json($data);
    }

    public function commmonFeedbackCharts(Request $request)
    {
        $args = $this->dashboardService->processFilters($request);
        $sql = $this->getCoreSqlBeneficiary($args);
        $sql->groupBy('beneficiary_import.district_id');
        $sql->select(
            'city_master.name',
            DB::raw('COUNT(beneficiary_import.id) as totalCalls'),
            DB::raw('COUNT(CASE WHEN feedback.is_compliant = 0 THEN feedback.id END) as feedbackCount'),
        );
        $feedbackCounts = $sql->get();
        return $feedbackCounts;
    }

    public function getMaximumFeedbackChart(Request $request)
    {
        $feedbackCounts = $this->commmonFeedbackCharts($request);
        $data = new AppClass();
        $data->title = 'Maximum feedback received districts';
        $data->renderId = 'maximum-feedback-chart';
        $data->yaxisTitle = 'Calls';

        if ($feedbackCounts) {

            $sorted = $feedbackCounts->sortBy([
                ['feedbackCount', 'desc'],
            ])->values();

            $data->xaxisColumns = $sorted->pluck('name');
            $data->series = new Collection();

            $total = new AppClass();
            $total->name = 'Total Calls';
            $total->data = $sorted->pluck('totalCalls');
            $data->series->add($total);

            $feedback = new AppClass();
            $feedback->name = 'Feedbacks';
            $feedback->data = $sorted->pluck('feedbackCount');
            $data->series->add($feedback);
        }

        return response()->json($data);
    }

    public function getMaximumComplaintsChartPercentage(Request $request)
    {
        $complaintsCounts = $this->commonComplaintsChart($request);
        $data = new AppClass();
        $data->title = 'Maximum Complaints (Percentage)';
        $data->renderId = 'maximum-complaints-chart-percentage';
        $data->yaxisTitle = 'Percentage';

        if ($complaintsCounts) {

            foreach ($complaintsCounts as $complaint) {
                $complaint->complaintPercentage = calcPercentage($complaint->complaintCount, $complaint->totalCalls);
            }

            $sorted = $complaintsCounts->sortBy([
                ['complaintPercentage', 'desc'],
            ])->values();

            $data->xaxisColumns = $sorted->pluck('name');
            $data->series = new Collection();

            $complaints = new AppClass();
            $complaints->name = 'Complaints Percentage';
            $complaints->data = $sorted->pluck('complaintPercentage');
            $data->series->add($complaints);
        }

        return response()->json($data);
    }

    public function commonComplaintsChart(Request $request)
    {
        $args = $this->dashboardService->processFilters($request);
        $sql = $this->getCoreSqlBeneficiary($args);
        $sql->groupBy('beneficiary_import.district_id');
        $sql->select(
            'city_master.name',
            DB::raw('COUNT(beneficiary_import.id) as totalCalls'),
            DB::raw('COUNT(CASE WHEN feedback.is_compliant = 1 THEN feedback.id END) as complaintCount'),
        );
        if (!is_null(Auth::user()->role)) {
            $sql->where('feedback.is_show', 1);
        }
        $complaintsCounts = $sql->get();
        return $complaintsCounts;
    }

    public function getMaximumComplaintsChart(Request $request)
    {
        $complaintsCounts = $this->commonComplaintsChart($request);
        $data = new AppClass();
        $data->title = 'Maximum Complaints';
        $data->renderId = 'maximum-complaints-chart';
        $data->yaxisTitle = 'Calls';

        if ($complaintsCounts) {

            $sorted = $complaintsCounts->sortBy([
                ['complaintCount', 'desc'],
            ])->values();

            $data->xaxisColumns = $sorted->pluck('name');
            $data->series = new Collection();

            $complaints = new AppClass();
            $complaints->name = 'Complaints';
            $complaints->data = $sorted->pluck('complaintCount');
            $data->series->add($complaints);

            $total = new AppClass();
            $total->name = 'Total Calls';
            $total->data = $sorted->pluck('totalCalls');
            $data->series->add($total);
        }

        return response()->json($data);
    }

    public function topComplaintsDistrictsPercentage(Request $request)
    {
        $complaintsCounts = $this->commonComplaintsChartDistrict($request);
        $data = new AppClass();
        $data->title = 'Top 5 Complaints received districts (Percentage)';
        $data->renderId = 'top-complaints-districts-percentage';
        $data->yaxisTitle = 'Percentage';

        if ($complaintsCounts) {

            foreach ($complaintsCounts as $complaint) {
                $complaint->complaintPercentage = calcPercentage($complaint->complaintCount, $complaint->totalCalls);
            }

            $sorted = $complaintsCounts->sortBy([
                ['complaintPercentage', 'desc'],
            ])->values();

            $data->xaxisColumns = $sorted->pluck('name');
            $data->series = new Collection();

            $complaints = new AppClass();
            $complaints->name = 'Complaints Percentage';
            $complaints->data = $sorted->pluck('complaintPercentage');
            $data->series->add($complaints);
        }

        return response()->json($data);
    }

    public function commonComplaintsChartDistrict(Request $request)
    {
        $args = $this->dashboardService->processFilters($request);
        $sql = $this->getCoreSqlBeneficiary($args);
        $sql->groupBy('beneficiary_import.district_id');
        $sql->select(
            'city_master.name',
            DB::raw('COUNT(beneficiary_import.id) as totalCalls'),
            DB::raw('COUNT(CASE WHEN feedback.is_compliant = 1 THEN feedback.id END) as complaintCount'),
        );
        if (!is_null(Auth::user()->role)) {
            $sql->where('feedback.is_show', 1);
        }
        $sql->orderBy('complaintCount', 'desc');
        $complaintsCounts = $sql->take(5)->get();
        return $complaintsCounts;
    }

    public function topComplaintsDistricts(Request $request)
    {
        $complaintsCounts = $this->commonComplaintsChartDistrict($request);
        $data = new AppClass();
        $data->title = 'Top 5 Complaints received districts';
        $data->renderId = 'top-complaints-districts';
        $data->yaxisTitle = 'Calls';

        if ($complaintsCounts) {

            $data->xaxisColumns = $complaintsCounts->pluck('name');
            $data->series = new Collection();

            $complaints = new AppClass();
            $complaints->name = 'Complaints';
            $complaints->data = $complaintsCounts->pluck('complaintCount');
            $data->series->add($complaints);

            $total = new AppClass();
            $total->name = 'Total Calls';
            $total->data = $complaintsCounts->pluck('totalCalls');
            $data->series->add($total);
        }

        return response()->json($data);
    }

    public function coreCallSkipReasons(Request $request)
    {
        $args = $this->dashboardService->processFilters($request);
        $sql = $this->getCoreSqlBeneficiary($args);
        $sql->groupBy('beneficiary_import.district_id');
        $sql->select(
            'city_master.name',
            DB::raw('COUNT(beneficiary_import.id) as totalCalls'),
            DB::raw('COUNT(CASE WHEN beneficiary_import.skip_reason = ' . $request->skipReason . ' THEN beneficiary_import.id END) as count')
        );
        $data = $sql->get();
        $sorted = $data->sortBy([
            ['count', 'desc'],
        ])->values()->take(5);
        return $sorted;
    }

    public function maxWrongNumbersChartPercentage(Request $request)
    {
        $data = new AppClass();
        $data->title = 'Top 5 wrong numbers districts (Percentage)';
        $data->renderId = 'max-wrong-numbers-percentage';
        $data->yaxisTitle = 'Percentage';

        $request->skipReason = BeneficiaryImport::SKIP_REASON_WRONG_NUMBER;
        $skipReason = $this->coreCallSkipReasons($request);

        if ($skipReason) {

            foreach ($skipReason as $complaint) {
                $complaint->percentage = calcPercentage($complaint->count, $complaint->totalCalls);
            }

            $sorted = $skipReason->sortBy([
                ['complaintPercentage', 'desc'],
            ])->values();

            $data->xaxisColumns = $sorted->pluck('name');
            $data->series = new Collection();

            $complaints = new AppClass();
            $complaints->name = 'Wrong Number Percentage';
            $complaints->data = $sorted->pluck('percentage');
            $data->series->add($complaints);
        }

        return response()->json($data);
    }

    public function maxWrongNumbersChart(Request $request)
    {
        $data = new AppClass();
        $data->title = 'Top 5 wrong numbers districts';
        $data->renderId = 'max-wrong-numbers';
        $data->yaxisTitle = 'Calls';

        $request->skipReason = BeneficiaryImport::SKIP_REASON_WRONG_NUMBER;
        $skipReason = $this->coreCallSkipReasons($request);

        if ($skipReason) {

            $data->xaxisColumns = $skipReason->pluck('name');
            $data->series = new Collection();

            $complaints = new AppClass();
            $complaints->name = 'Wrong Number';
            $complaints->data = $skipReason->pluck('count');
            $data->series->add($complaints);

            $total = new AppClass();
            $total->name = 'Total Calls';
            $total->data = $skipReason->pluck('totalCalls');
            $data->series->add($total);
        }

        return response()->json($data);
    }

    public function notAttendChartPercentage(Request $request)
    {
        $data = new AppClass();
        $data->title = 'Top 5 Unattend Calls districts (Percentage)';
        $data->renderId = 'not-attend-percentage';
        $data->yaxisTitle = 'Percentage';

        $request->skipReason = BeneficiaryImport::SKIP_REASON_NOT_ATTEND;
        $skipReason = $this->coreCallSkipReasons($request);

        if ($skipReason) {

            foreach ($skipReason as $complaint) {
                $complaint->percentage = calcPercentage($complaint->count, $complaint->totalCalls);
            }

            $sorted = $skipReason->sortBy([
                ['percentage', 'desc'],
            ])->values();

            $data->xaxisColumns = $sorted->pluck('name');
            $data->series = new Collection();

            $feedback = new AppClass();
            $feedback->name = 'Unattend Calls Percentage';
            $feedback->data = $sorted->pluck('percentage');
            $data->series->add($feedback);
        }

        return response()->json($data);
    }

    public function notAttendChart(Request $request)
    {
        $data = new AppClass();
        $data->title = 'Top 5 Unattend Calls districts';
        $data->renderId = 'not-attend';
        $data->yaxisTitle = 'Calls';

        $request->skipReason = BeneficiaryImport::SKIP_REASON_NOT_ATTEND;
        $skipReason = $this->coreCallSkipReasons($request);

        if ($skipReason) {

            $data->xaxisColumns = $skipReason->pluck('name');
            $data->series = new Collection();

            $total = new AppClass();
            $total->name = 'Total Calls';
            $total->data = $skipReason->pluck('totalCalls');
            $data->series->add($total);

            $feedback = new AppClass();
            $feedback->name = 'Unattend Calls';
            $feedback->data = $skipReason->pluck('count');
            $data->series->add($feedback);
        }

        return response()->json($data);
    }

    public function getResolvedFeedbacks(Request $request)
    {
        $args = $this->dashboardService->processFilters($request);
        $sql = $this->getCoreSqlFeedBack($args);
        // $sql->where('feedback.resolved_status', Feedback::RESOLVED_STATUS_RESOLVED);
        $sql->where('feedback.is_compliant', Feedback::IS_COMPLIANT);
        $sql->groupBy('beneficiary_import.district_id');
        $sql->select(
            'city_master.name',
            DB::raw('COUNT(feedback.id) as complaintsCount'),
            DB::raw('COUNT(CASE WHEN feedback.resolved_status = 1 THEN feedback.id END) as count'),
        );
        $sql->orderBy('count', 'desc');
        $data = $sql->get();
        return $data;
    }

    public function resolvedChartPercentage(Request $request)
    {
        $data = new AppClass();
        $data->title = 'Resolved Lists (Percentage)';
        $data->renderId = 'resolved-charts-percentage';
        $data->yaxisTitle = 'Percentage';

        $feedback = $this->getResolvedFeedbacks($request);

        if ($feedback) {

            foreach ($feedback as $complaint) {
                $complaint->percentage = calcPercentage($complaint->count, $complaint->complaintsCount);
            }

            $sorted = $feedback->sortBy([
                ['percentage', 'desc'],
            ])->values();

            $data->xaxisColumns = $sorted->pluck('name');
            $data->series = new Collection();

            $total = new AppClass();
            $total->name = 'Resolved Percentage';
            $total->data = $sorted->pluck('percentage');
            $data->series->add($total);
        }

        return response()->json($data);
    }

    public function resolvedChart(Request $request)
    {
        $data = new AppClass();
        $data->title = 'Resolved Lists';
        $data->renderId = 'resolved-charts';
        $data->yaxisTitle = 'Counts';

        $feedback = $this->getResolvedFeedbacks($request);

        if ($feedback) {

            $data->xaxisColumns = $feedback->pluck('name');
            $data->series = new Collection();

            $total = new AppClass();
            $total->name = 'Complaints';
            $total->data = $feedback->pluck('complaintsCount');
            $data->series->add($total);

            $total = new AppClass();
            $total->name = 'Resolved';
            $total->data = $feedback->pluck('count');
            $data->series->add($total);
        }

        return response()->json($data);
    }

    public function reconnectData(Request $request)
    {
        $args = $this->dashboardService->processFilters($request);
        $sql = $this->getCoreSqlBeneficiary($args);
        $sql->where('beneficiary_import.reconnected', 1);
        $sql->groupBy('beneficiary_import.district_id');
        if (!is_null(Auth::user()->role)) {
            $sql->where('feedback.is_show', 1);
        }
        $sql->select(
            'city_master.name',
            DB::raw('COUNT(beneficiary_import.id) as totalCalls'),
            DB::raw('COUNT(CASE WHEN feedback.is_compliant = 1 THEN feedback.id END) as complaintCount'),
            DB::raw('COUNT(CASE WHEN feedback.is_compliant = 0 THEN feedback.id END) as feedbackCount'),
        );
        $sql->orderBy('complaintCount', 'desc');
        $reconnect = $sql->get();
        return $reconnect;
    }

    public function getComplaints(Request $request)
    {
        $args = $this->dashboardService->processFilters($request);
        $sql = $this->getCoreSqlFeedBackComparision($args);
        if ($request->complaints) {
            $sql->where('feedback.is_compliant', Feedback::IS_COMPLIANT);
        } else {
            $sql->where('feedback.is_compliant', 0);
        }

        $sql->groupBy('beneficiary_import.district_id');
        $sql->groupBy(DB::raw('MONTH(feedback.created_at)'));
        $sql->select(
            'city_master.name',
            DB::raw('COUNT(feedback.id) as count'),
            DB::raw('SUBSTRING(feedback.created_at,6,2) as month'),
            DB::raw('MONTHNAME(feedback.created_at) as monthName'),
        );
        $data = $sql->get();
        return $data;
    }

    public function makeDataMonthwise($data)
    {
        $complaints = new Collection();

        foreach ($data as $row) {
            foreach ($row->month as $month) {
                foreach ($month as $record) {

                    if (!isset($complaints[$record->monthName])) {
                        $complaints[$record->monthName] = new Collection();
                    }
                    $complaints[$record->monthName]->add($record);
                }
            }
        }

        return $complaints;
    }

    public function makeDataMonthwiseTable($data, $column)
    {
        $complaints = new Collection();

        foreach ($data as $row) {
            if (!isset($complaints[$row->name])) {
                $complaints[$row->name] = new AppClass();
                $complaints[$row->name]->month = new Collection();
                $complaints[$row->name]->district = $row->name;

                foreach ($column as $header) {
                    $complaints[$row->name]->month[$header] = new Collection();
                }
            }

            $complaints[$row->name]->month[$row->monthName]->add($row);
        }

        $complaints = $this->fillEmptyMonths($complaints, $column);

        return $complaints;
    }

    public function fillEmptyMonths($complaints, $column)
    {
        foreach ($complaints as $key => $complaint) {
            foreach ($complaint->month as $key1 => $complaintRow) {

                if (count($complaintRow) == 0) {
                    $node = new AppClass();
                    $node->count = 0;
                    $node->monthName = $key1;
                    $node->name = $key;
                    $complaintRow->add($node);
                }
            }
        }
        return $complaints;
    }

    public function complaintsComparision(Request $request)
    {
        $data = new AppClass();
        $data->title = 'Complaints comparison last 3 months';
        $data->renderId = 'complaints-comparision';
        $data->yaxisTitle = 'Complaints';
        $request->complaints = true;
        $complaints = $this->getComplaints($request);
        $column = $complaints->pluck('monthName')->unique()->values();
        $monthwiseComplaints = $this->makeDataMonthwiseTable($complaints, $column);
        $monthwiseComplaints = $this->makeDataMonthwise($monthwiseComplaints);

        if ($monthwiseComplaints) {

            $data->xaxisColumns = $complaints->pluck('name')->unique()->values();
            $data->series = new Collection();
            foreach ($monthwiseComplaints as $key => $monthwise) {
                $total = new AppClass();
                $total->name = $key;
                $total->data = $monthwise->pluck('count');
                $data->series->add($total);
            }
        }

        return response()->json($data);
    }

    public function feedbackComparision(Request $request)
    {
        $data = new AppClass();
        $data->title = 'Feedback comparison last 3 months';
        $data->renderId = 'feedback-comparision';
        $data->yaxisTitle = 'feedback';
        $request->complaints = false;
        $complaints = $this->getComplaints($request);
        $column = $complaints->pluck('monthName')->unique()->values();
        $monthwiseComplaints = $this->makeDataMonthwiseTable($complaints, $column);
        $monthwiseComplaints = $this->makeDataMonthwise($monthwiseComplaints);

        if ($monthwiseComplaints) {

            $data->xaxisColumns = $complaints->pluck('name')->unique()->values();
            $data->series = new Collection();

            foreach ($monthwiseComplaints as $key => $monthwise) {
                $total = new AppClass();
                $total->name = $key;
                $total->data = $monthwise->pluck('count');
                $data->series->add($total);
            }
        }

        return response()->json($data);
    }

    public function complaintsComparisionTable(Request $request)
    {
        $request->complaints = true;
        $complaints = $this->getComplaints($request);
        $column = $complaints->pluck('monthName')->unique()->values();
        $monthwiseComplaints = $this->makeDataMonthwiseTable($complaints, $column);
        $data = [
            'rows' => $monthwiseComplaints,
            'column' => $column
        ];
        return response()->json($data);
    }

    public function feedbackComparisionTable(Request $request)
    {
        $request->complaints = false;
        $complaints = $this->getComplaints($request);
        $column = $complaints->pluck('monthName')->unique()->values();
        $monthwiseComplaints = $this->makeDataMonthwiseTable($complaints, $column);
        $data = [
            'rows' => $monthwiseComplaints,
            'column' => $column
        ];
        return response()->json($data);
    }

    public function getReconnectedChartPercentage(Request $request)
    {
        $data = new AppClass();
        $data->title = 'Reconnected Percentage';
        $data->renderId = 'reconnected-charts-percentage';
        $data->yaxisTitle = 'Percentage';

        $reconnect = $this->reconnectData($request);

        if ($reconnect) {

            foreach ($reconnect as $complaint) {
                $complaint->feedbackPercentage = calcPercentage($complaint->feedbackCount, $complaint->totalCalls);
                $complaint->complaintsPercentage = calcPercentage($complaint->complaintCount, $complaint->totalCalls);
            }

            $sorted = $reconnect->sortBy([
                ['complaintsPercentage', 'desc'],
            ])->values();

            $data->xaxisColumns = $sorted->pluck('name');
            $data->series = new Collection();

            $total = new AppClass();
            $total->name = 'Feedback Percentage';
            $total->data = $sorted->pluck('feedbackPercentage');
            $data->series->add($total);

            $total = new AppClass();
            $total->name = 'Complaints Percentage';
            $total->data = $sorted->pluck('complaintsPercentage');
            $data->series->add($total);
        }

        return response()->json($data);
    }

    public function getReconnectedChart(Request $request)
    {
        $data = new AppClass();
        $data->title = 'Reconnected';
        $data->renderId = 'reconnected-charts';
        $data->yaxisTitle = 'Calls';

        $reconnect = $this->reconnectData($request);
        if ($reconnect) {

            $data->xaxisColumns = $reconnect->pluck('name');
            $data->series = new Collection();

            $total = new AppClass();
            $total->name = 'Total Calls';
            $total->data = $reconnect->pluck('totalCalls');
            $data->series->add($total);

            $total = new AppClass();
            $total->name = 'Feedback';
            $total->data = $reconnect->pluck('feedbackCount');
            $data->series->add($total);

            $total = new AppClass();
            $total->name = 'Complaints';
            $total->data = $reconnect->pluck('complaintCount');
            $data->series->add($total);
        }

        return response()->json($data);
    }

    public function getProgramwiseChartPercentage(Request $request)
    {
        $programCounts = $this->commonProgramChartData($request);

        $data = new AppClass();
        $data->title = 'Program wise Calls Made (Percentage)';
        $data->renderId = 'programwise-chart-percentage';
        $data->yaxisTitle = 'Percentage';
        if ($programCounts) {

            foreach ($programCounts as $program) {
                $program->complainPercentage = calcPercentage($program->complaintCount, $program->totalCalls);
                $program->feedbackPercentage = calcPercentage($program->feedbackCount, $program->totalCalls);
            }

            $data->xaxisColumns = $programCounts->pluck('name');
            $data->series = new Collection();

            $feedback = new AppClass();
            $feedback->name = 'Feedbacks';
            $feedback->data = $programCounts->pluck('feedbackPercentage');
            $data->series->add($feedback);

            $compliant = new AppClass();
            $compliant->name = 'Complaints';
            $compliant->data = $programCounts->pluck('complainPercentage');
            $data->series->add($compliant);
        }

        return response()->json($data);
    }

    public function commonProgramChartData(Request $request)
    {
        $args = $this->dashboardService->processFilters($request);
        $sql = $this->getCoreSqlBeneficiary($args);
        if (!is_null(Auth::user()->role)) {
            $sql->where('feedback.is_show', 1);
        }
        $sql->groupBy('beneficiary_import.program_id');
        $sql->select(
            'program_master.name',
            DB::raw('COUNT(beneficiary_import.id) as totalCalls'),
            DB::raw('COUNT(CASE WHEN feedback.is_compliant = 1 THEN feedback.id END) as complaintCount'),
            DB::raw('COUNT(CASE WHEN feedback.is_compliant = 0 THEN feedback.id END) as feedbackCount'),
        );
        $sql->orderBy('complaintCount', 'desc');
        $programCounts = $sql->get();
        return $programCounts;
    }

    public function getProgramwiseChart(Request $request)
    {
        $programCounts = $this->commonProgramChartData($request);
        $data = new AppClass();
        $data->title = 'Program wise Calls Made';
        $data->renderId = 'programwise-chart';
        $data->yaxisTitle = 'Calls';
        if ($programCounts) {

            $data->xaxisColumns = $programCounts->pluck('name');
            $data->series = new Collection();

            $total = new AppClass();
            $total->name = 'Total Calls';
            $total->data = $programCounts->pluck('totalCalls');
            $data->series->add($total);

            $feedback = new AppClass();
            $feedback->name = 'Feedbacks';
            $feedback->data = $programCounts->pluck('feedbackCount');
            $data->series->add($feedback);

            $compliant = new AppClass();
            $compliant->name = 'Complaints';
            $compliant->data = $programCounts->pluck('complaintCount');
            $data->series->add($compliant);
        }

        return response()->json($data);
    }

    public function getCoreSql($args)
    {
        $sql = DB::table('beneficiary_import')
            ->leftJoin('feedback', 'feedback.beneficiary_ref_id', 'beneficiary_import.id')
            ->join('program_master', 'program_master.id', 'beneficiary_import.program_id')
            ->join('city_master', 'city_master.id', 'beneficiary_import.district_id')
            ->join('block_master', 'block_master.id', 'beneficiary_import.block_id');

        $sql->where('beneficiary_import.status', '!=', BeneficiaryImport::STATUS_ACTIVE);

        if ($args->scheme != 'All') {
            $sql->where('beneficiary_import.scheme_id', $args->scheme);
        }

        if ($args->program != 'All') {
            $sql->where('beneficiary_import.program_id', $args->program);
        }

        if ($args->district != 'All') {
            $sql->where('beneficiary_import.district_id', $args->district);
        }

        if ($args->block != 'All') {
            $sql->where('beneficiary_import.block_id', $args->block);
        }

        return $sql;
    }


    public function getCoreSqlFeedBack($args)
    {
        $sql = $this->getCoreSql($args);
        $sql->where('feedback.status', Feedback::STATUS_ACTIVE)
            ->whereDate('feedback.created_at', '>=', $args->fromDate)
            ->whereDate('feedback.created_at', '<=', $args->toDate);
        if (!is_null(Auth::user()->role)) {
            $sql->where('feedback.is_show', 1);
        }
        return $sql;
    }

    public function getCoreSqlFeedBackComparision($args)
    {

        $fromDate = Carbon::parse($args->fromDate)->subMonths(2)->format('Y-m-01');
        $toDate = Carbon::parse($args->fromDate)->format('Y-m-t');
        $sql = $this->getCoreSql($args);
        $sql->where('feedback.status', Feedback::STATUS_ACTIVE)
            ->whereDate('feedback.created_at', '>=', $fromDate)
            ->whereDate('feedback.created_at', '<=', $toDate);
        if (!is_null(Auth::user()->role)) {
            $sql->where('feedback.is_show', 1);
        }
        return $sql;
    }

    public function getCoreSqlBeneficiary($args)
    {
        $sql = $this->getCoreSql($args);
        $sql->whereDate('beneficiary_import.updated_at', '>=', $args->fromDate)
            ->whereDate('beneficiary_import.updated_at', '<=', $args->toDate);
        return $sql;
    }

    public function getEscDashboardCounts(Request $request)
    {
        $args = $this->dashboardService->processFilters($request);
        $args->type = null;
        $data = $this->getDashboardCounts($args);
        $html = view('reports.escalation-dashboard-details', compact('data', 'args'))->render();
        $response = [
            'status' => true,
            'redirect' => null,
            'renderId' => '#esc-dashboard-content',
            'html' => $html
        ];
        return response()->json($response);
    }

    public function getDashboardCounts($args)
    {
        $data = $this->presentationEntities->getEscDashboardEntity();
        foreach ($data as $escalation) {
            $count = $this->getEscEntityCounts($escalation, $args);
            if (isset($count)) {
                $escalation->count = $count->count;
            }
        }
        return $data;
    }

    public function getCoreSqlEsc($args)
    {
        $sql = DB::table('feedback')
            ->join('beneficiary_import', 'beneficiary_import.id', 'feedback.beneficiary_ref_id')
            ->where('feedback.status', Feedback::STATUS_ACTIVE)
            ->where('feedback.is_compliant', Feedback::IS_COMPLIANT)
            ->whereDate('feedback.created_at', '>=', $args->fromDate)
            ->whereDate('feedback.created_at', '<=', $args->toDate);

        if ($args->scheme != 'All') {
            $sql->where('beneficiary_import.scheme_id', $args->scheme);
        }

        if ($args->program != 'All') {
            $sql->where('beneficiary_import.program_id', $args->program);
        }

        if ($args->district != 'All') {
            $sql->where('beneficiary_import.district_id', $args->district);
        }

        if ($args->block != 'All') {
            $sql->where('beneficiary_import.block_id', $args->block);
        }

        if (!is_null(Auth::user()->role)) {
            $sql->where('feedback.is_show', 1);
        }
        return $sql;
    }

    public function getEscEntityCounts($escalation, $args)
    {
        $sql = $this->getCoreSqlEsc($args);

        if ($escalation->title == PresentationEntities::AVG_TIME_TO_RESOLVE) {
            $sql->where('feedback.resolved_status', Feedback::RESOLVED_STATUS_RESOLVED);
            $sql->select(DB::raw('DATEDIFF(feedback.resolved_at, feedback.created_at) as diff'));
            $dateDiff = $sql->get();

            $resolve = new AppClass();
            $resolve->count = '0 Day';

            if ($dateDiff) {
                $sum = $dateDiff->sum('diff');
                $totalCount =  $dateDiff->count();
                if ($sum > 0) {
                    $avg = round($sum / $totalCount, 1);
                    $resolve->count = $avg . ' Days';
                }
            }
            return $resolve;
        }

        $sql->select(DB::raw('COUNT(feedback.id) as count'));
        if ($escalation->title == PresentationEntities::WORKER_RESOLVED) {
            $sql->where('feedback.resolved_status', Feedback::RESOLVED_STATUS_RESOLVED);
            $sql->whereRaw('(feedback.resolved_level = 1 OR feedback.resolved_level IS NULL)');
        } else if ($escalation->title == PresentationEntities::CDPO_RESOLVED) {
            $sql->where('feedback.resolved_status', Feedback::RESOLVED_STATUS_RESOLVED);
            $sql->where('feedback.resolved_level', 2);
        } else if ($escalation->title == PresentationEntities::DPO_RESOLVED) {
            $sql->where('feedback.resolved_status', Feedback::RESOLVED_STATUS_RESOLVED);
            $sql->where('feedback.resolved_level', 3);
        } else if ($escalation->title == PresentationEntities::PENDING_TO_RESOLVE) {
            $sql->where('feedback.resolved_status', Feedback::RESOLVED_STATUS_PENDING);
        }

        if ($args->type == 'detailed-report') {
            $sql->leftJoin('city_master', 'city_master.id', 'beneficiary_import.district_id');
            $sql->leftJoin('program_master', 'program_master.id', 'beneficiary_import.program_id');
            $sql->leftJoin('scheme_master', 'scheme_master.id', 'beneficiary_import.scheme_id');
            $sql->leftJoin('block_master', 'block_master.id', 'beneficiary_import.block_id');
            $sql->leftJoin('awc_master', 'awc_master.id', 'beneficiary_import.awc');
            $sql->join('users', 'users.id', 'feedback.created_by');
            $sql->select(
                'beneficiary_import.parent_name',
                'feedback.created_at',
                'scheme_master.name as scheme_name',
                'program_master.name as program_name',
                'city_master.name as district_name',
                'block_master.name as block_name',
                'beneficiary_import.mobile_number',
                'awc_master.name as awc_name',
                'beneficiary_import.status',
                'beneficiary_import.id',
                'users.name as created_user',
                'beneficiary_import.awc as awc_id',
                'feedback.feedback_no'
            );
            return $sql;
        }

        return $sql->first();
    }

    public function getDetailedReport(Request $request)
    {
        $args = new AppClass();
        $args->title = $request->refId;
        $args->fromDate = $request->fromDate;
        $args->toDate = $request->toDate;
        $args->scheme = $request->scheme;
        $args->program = $request->program;
        $args->district = $request->district;
        $args->block = $request->block;
        $args->type = 'detailed-report';
        return $this->getEscEntityCounts($args, $args);
    }

    public function complaintsResolvedByWorker(Request $request)
    {
        $args = $this->dashboardService->processFilters($request);
        $args->title = null;
        $complaints = $this->getWorkerwiseComplaints($args);

        $data = new AppClass();
        $data->title = 'Complaints Resolved by worker districtwise';
        $data->renderId = 'complaints-resolved-by-worker';
        $data->yaxisTitle = 'Complaints';

        if ($complaints) {

            $data->xaxisColumns = $complaints->pluck('districtName');
            $data->series = new Collection();

            $total = new AppClass();
            $total->name = 'Total';
            $total->data = $complaints->pluck('complaintCount');
            $data->series->add($total);

            $compliant = new AppClass();
            $compliant->name = 'Pending';
            $compliant->data = $complaints->pluck('pendingResolved');
            $data->series->add($compliant);

            $feedback = new AppClass();
            $feedback->name = 'Worker';
            $feedback->data = $complaints->pluck('workerResolved');
            $data->series->add($feedback);

            $compliant = new AppClass();
            $compliant->name = 'CDPO';
            $compliant->data = $complaints->pluck('cdpoResolved');
            $data->series->add($compliant);

            $compliant = new AppClass();
            $compliant->name = 'DPO';
            $compliant->data = $complaints->pluck('dpoResolved');
            $data->series->add($compliant);
        }

        return response()->json($data);
    }

    public function getWorkerwiseComplaints($args)
    {
        $sql = $this->getCoreSqlEsc($args);
        $sql->join('city_master', 'city_master.id', 'beneficiary_import.district_id');
        $sql->groupBy('beneficiary_import.district_id');
        $sql->select(
            'city_master.name as districtName',
            DB::raw('COUNT(feedback.id) as complaintCount'),
            DB::raw('COUNT(CASE WHEN feedback.resolved_status = 0 THEN feedback.id END) as pendingResolved'),
            DB::raw('COUNT(CASE WHEN (feedback.resolved_level = 1 OR feedback.resolved_level IS NULL) AND feedback.resolved_status = 1 THEN feedback.id END) as workerResolved'),
            DB::raw('COUNT(CASE WHEN feedback.resolved_level = 2 AND feedback.resolved_status = 1 THEN feedback.id END) as cdpoResolved'),
            DB::raw('COUNT(CASE WHEN feedback.resolved_level = 3 AND feedback.resolved_status = 1 THEN feedback.id END) as dpoResolved')
        );
        $complaint = $sql->get();
        return $complaint;
    }

    public function districtwiseAvgTimeResolved(Request $request)
    {
        $args = $this->dashboardService->processFilters($request);
        $sql = $this->getCoreSqlEsc($args);
        $sql->join('city_master', 'city_master.id', 'beneficiary_import.district_id');
        $sql->where('feedback.resolved_status', Feedback::RESOLVED_STATUS_RESOLVED);
        $sql->select(
            'city_master.name as districtName',
            DB::raw('DATEDIFF(feedback.resolved_at, feedback.created_at) as diff'),
        );
        $resolved = $sql->get();
        $grouped = $resolved->groupBy('districtName');
        $data = $this->calAvgTimeDistrictwise($grouped);
        return response()->json($data);
    }

    public function calAvgTimeDistrictwise($complaints)
    {
        $data = new Collection();

        foreach ($complaints as $complaint) {
            $items = new AppClass();
            $items->resolvedComplaints = 0;
            $days = 0;
            foreach ($complaint as $value) {
                $items->districtName = $value->districtName;
                $days += $value->diff;
                $items->resolvedComplaints++;
            }
            $items->diff = round($days / count($complaint), 2);
            $data->add($items);
        }
        return $data;
    }
}
