<?php

namespace App\Services\Reports;

use App\Models\Feedback;
use App\Models\Masters\BlockMaster;
use App\Models\Masters\CityMaster;
use App\Models\Masters\ProgramMaster;
use App\Models\Masters\SchemeMaster;
use App\Services\DashboardService;
use App\Utilities\AppClass;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use DB;
use Auth;

class EscalationDashboardService
{
    const COMPLAINTS_RECEIVED = 'Complaints Received';
    const WORKER_RESOLVED = 'Worker Resolved';
    const CDPO_RESOLVED = 'CDPO Resolved';
    const DPO_RESOLVED = 'DPO Resolved';
    const PENDING_TO_RESOLVE = 'Pending to Resolve';
    const AVG_TIME_TO_RESOLVE = 'Avg time to Resolve';

    private $dashboardService;
    public function __construct(DashboardService $dashboardService)
    {
        $this->dashboardService = $dashboardService;
    }

    public function getEscalationDashboard(Request $request)
    {
        $data = new AppClass();
        $data->view = 'reports.escalation-dashboard';
        $data->scheme = SchemeMaster::whereStatus(SchemeMaster::STATUS_ACTIVE)->get();
        $data->program = ProgramMaster::whereStatus(ProgramMaster::STATUS_ACTIVE)->get();
        $data->district = CityMaster::whereStatus(CityMaster::STATUS_ACTIVE)->get();
        return $data;
    }

    public function getDashboardEntity()
    {
        $data = new Collection();

        $items = new AppClass();
        $items->title = self::COMPLAINTS_RECEIVED;
        $items->count = 0;
        $items->class = 'fa fa-bar-chart text-success';
        $items->showLink = true;
        $data->add($items);

        $items = new AppClass();
        $items->count = 0;
        $items->title = self::WORKER_RESOLVED;
        $items->class = 'fa fa-bar-chart text-success';
        $items->showLink = true;
        $data->add($items);

        $items = new AppClass();
        $items->count = 0;
        $items->title = self::CDPO_RESOLVED;
        $items->class = 'fa fa-bar-chart text-success';
        $items->showLink = true;
        $data->add($items);

        $items = new AppClass();
        $items->count = 0;
        $items->title = self::DPO_RESOLVED;
        $items->class = 'fa fa-bar-chart text-success';
        $items->showLink = true;
        $data->add($items);

        $items = new AppClass();
        $items->count = 0;
        $items->title = self::PENDING_TO_RESOLVE;
        $items->class = 'fa fa-bar-chart text-success';
        $items->showLink = true;
        $data->add($items);

        $items = new AppClass();
        $items->count = 0;
        $items->title = self::AVG_TIME_TO_RESOLVE;
        $items->class = 'fa fa-bar-chart text-success';
        $items->showLink = false;
        $data->add($items);

        return $data;
    }

    public function getDashboardData(Request $request)
    {
        $args = $this->dashboardService->processFilters($request);
        $args->type = null;
        $data = $this->getDashboardCounts($args);
        $html = view('reports.escalation-dashboard-details', compact('data', 'args'))->render();
        $response = [
            'status' => true,
            'redirect' => null,
            'renderId' => '#dashboard-content',
            'html' => $html
        ];
        return response()->json($response);
    }

    public function getDashboardCounts($args)
    {
        $data = $this->getDashboardEntity();
        foreach ($data as $escalation) {
            $count = $this->getEntityCounts($escalation, $args);
            if (isset($count)) {
                $escalation->count = $count->count;
            }
        }
        return $data;
    }

    public function getCoreSql($args)
    {
        $sql = DB::table('feedback')
            ->join('beneficiary_import', 'beneficiary_import.id', 'feedback.beneficiary_ref_id')
            ->where('feedback.status', Feedback::STATUS_ACTIVE)
            ->where('feedback.is_compliant', Feedback::IS_COMPLIANT)
            ->whereDate('feedback.created_at', '>=', $args->fromDate)
            ->whereDate('feedback.created_at', '<=', $args->toDate);

        if ($args->scheme != 'All') {
            $sql->where('beneficiary_import.scheme_id', $args->scheme);
        }

        if ($args->program != 'All') {
            $sql->where('beneficiary_import.program_id', $args->program);
        }

        if ($args->district != 'All') {
            $sql->where('beneficiary_import.district_id', $args->district);
        }

        if (!is_null(Auth::user()->role)) {
            $sql->where('feedback.is_show', 1);
        }
        return $sql;
    }

    public function getEntityCounts($escalation, $args)
    {
        $sql = $this->getCoreSql($args);

        if ($escalation->title == self::AVG_TIME_TO_RESOLVE) {
            $sql->where('feedback.resolved_status', Feedback::RESOLVED_STATUS_RESOLVED);
            $sql->select(DB::raw('DATEDIFF(feedback.resolved_at, feedback.created_at) as diff'));
            $dateDiff = $sql->get();

            $resolve = new AppClass();
            $resolve->count = '0 Day';

            if ($dateDiff) {
                $sum = $dateDiff->sum('diff');
                $totalCount =  $dateDiff->count();
                if ($sum > 0) {
                    $avg = round($sum / $totalCount, 1);
                    $resolve->count = $avg . ' Days';
                }
            }
            return $resolve;
        }

        $sql->select(DB::raw('COUNT(feedback.id) as count'));
        if ($escalation->title == self::WORKER_RESOLVED) {
            $sql->where('feedback.resolved_status', Feedback::RESOLVED_STATUS_RESOLVED);
            $sql->whereRaw('(feedback.resolved_level = 1 OR feedback.resolved_level IS NULL)');
        } else if ($escalation->title == self::CDPO_RESOLVED) {
            $sql->where('feedback.resolved_status', Feedback::RESOLVED_STATUS_RESOLVED);
            $sql->where('feedback.resolved_level', 2);
        } else if ($escalation->title == self::DPO_RESOLVED) {
            $sql->where('feedback.resolved_status', Feedback::RESOLVED_STATUS_RESOLVED);
            $sql->where('feedback.resolved_level', 3);
        } else if ($escalation->title == self::PENDING_TO_RESOLVE) {
            $sql->where('feedback.resolved_status', Feedback::RESOLVED_STATUS_PENDING);
        }

        if ($args->type == 'detailed-report') {
            $sql->leftJoin('city_master', 'city_master.id', 'beneficiary_import.district_id');
            $sql->leftJoin('program_master', 'program_master.id', 'beneficiary_import.program_id');
            $sql->leftJoin('scheme_master', 'scheme_master.id', 'beneficiary_import.scheme_id');
            $sql->join('users', 'users.id', 'feedback.created_by');
            $sql->select(
                'beneficiary_import.customer_name',
                'feedback.created_at',
                'scheme_master.name as scheme_name',
                'program_master.name as program_name',
                'city_master.name as district_name',
                'beneficiary_import.mobile_number',
                'beneficiary_import.status',
                'beneficiary_import.id',
                'users.name as created_user',
                'beneficiary_import.awc as awc_id',
                'feedback.feedback_no'
            );
            return $sql;
        }

        return $sql->first();
    }

    public function getDetailedReport(Request $request)
    {
        $args = new AppClass();
        $args->title = $request->refId;
        $args->fromDate = $request->fromDate;
        $args->toDate = $request->toDate;
        $args->scheme = $request->scheme;
        $args->program = $request->program;
        $args->district = $request->district;
        $args->block = $request->block;
        $args->type = 'detailed-report';
        return $this->getEntityCounts($args, $args);
    }
}
