<?php

namespace App\Services\Reports;

use App\Models\Masters\AgentMaster;
use App\Models\Masters\CallSkipReason;
use App\Services\DashboardService;
use App\Services\ExcelExportService;
use App\Services\Reports\ReportService;
use Carbon\Carbon;
use Illuminate\Http\Request;
use DB;
use Illuminate\Support\Collection;
use stdClass;

class EmployeeSummaryService
{
    private $reportService;
    private $excelExportService;
    private $dashboardService;

    public function __construct(ReportService $reportService, ExcelExportService $excelExportService, DashboardService $dashboardService)
    {
        $this->reportService = $reportService;
        $this->excelExportService = $excelExportService;
        $this->dashboardService = $dashboardService;
    }

    function querySql1($args)
    {
        $newArgs = clone $args;
        $sql = $this->dashboardService->coreAgentAssignFollowDetails($newArgs);
        $sql->select(
            'beneficiary_import.branch_id',
            'beneficiary_import.agent_id',
            DB::raw('COUNT(beneficiary_import.id) as allocatedCalls'),
        )
            ->groupBy('beneficiary_import.branch_id', 'beneficiary_import.agent_id');
        $data = $sql->get();
        return $data;
    }

    function querySql3($args)
    {
        $newArgs = clone $args;
        $sql = $this->dashboardService->coreAgentCallDetails($newArgs);
        $sql->select(
            'beneficiary_import.branch_id',
            'beneficiary_import.agent_id',
            DB::raw('COUNT(DISTINCT feedback.id) as dialledCalls'),
            DB::raw('COUNT(DISTINCT CASE WHEN feedback.call_status = 1 THEN feedback.id END) as connectedCalls'),
            DB::raw('COUNT(DISTINCT CASE WHEN feedback.call_status = 2 THEN feedback.id END) as notConnectedCalls'),

        )
            ->groupBy('beneficiary_import.branch_id', 'beneficiary_import.agent_id');

        $data = $sql->get();
        return $data;
    }

    function querySql5($args)
    {
        $newArgs = clone $args;
        $sql = $this->dashboardService->coreAgentCallDetails($newArgs);
        $sql->select(
            'beneficiary_import.branch_id',
            'call_skip_reason.id',
            'call_skip_reason.name',
            'beneficiary_import.agent_id',
            DB::raw('COUNT(feedback.id) as count')
        )
            ->leftJoin('call_skip_reason', 'call_skip_reason.id', 'feedback.call_skip_reason')
            ->groupBy('beneficiary_import.branch_id', 'call_skip_reason.id','beneficiary_import.agent_id');
        $data = $sql->get();
        return $data;
    }

    public function getData(Request $request)
    {
        $args = $this->reportService->processrequestFilters($request);
        $fromDate = Carbon::parse($args->years . '-' . $args->months . '-01')->format('Y-m-d');
        $toDate = Carbon::parse($fromDate)->endOfMonth()->format('Y-m-d');
        $args->fromDate = $fromDate;
        $args->toDate = $toDate;

        if ($args->branch == '') {
            $args->branch = 'All';
        }
        if ($args->agent == '') {
            $args->agent = 'All';
        }

        //getting all call skip reasons
        $callReason = CallSkipReason::whereStatus(CallSkipReason::STATUS_ACTIVE)->get();

        //getting branches
        $agentSql = AgentMaster::select('agent_master.id', 'agent_master.name', 'branch_master.name as branchName')
            ->join('branch_master', 'branch_master.id', 'agent_master.branch_id')
            ->where('agent_master.status', AgentMaster::STATUS_ACTIVE);
        if ((int) $args->branch > 0) {
            $agentSql->where('branch_master.id', $args->branch);
        }
        if ((int) $args->agent > 0) {
            $agentSql->where('agent_master.id', $args->agent);
        }
        $agentSql->orderBy('branch_master.name');
        $agentList = $agentSql->get();

        $queryResult1 = $this->querySql1($args);
        $queryResult3 = $this->querySql3($args);
        $queryResult5 = $this->querySql5($args);

        $result = new stdClass;
        $result->records = new Collection();
        $result->title = 'Telecalling Report' . AppDateFormat($fromDate) . ' - ' . AppDateFormat($toDate);
        $result->connected = $callReason->where('is_answered', 1);
        $result->notConnected = $callReason->where('is_answered', '!=', 1);

        foreach ($agentList as $agent) {

            $item = new stdClass;

            $queryAgentData1 = $queryResult1->where('agent_id', $agent->id)->first();
            $queryAgentData3 = $queryResult3->where('agent_id', $agent->id)->first();

            $allocatedCalls = $queryAgentData1->allocatedCalls ?? 0;
            $dialledCalls = $queryAgentData3->dialledCalls ?? 0;
            $connectedCalls = $queryAgentData3->connectedCalls ?? 0;
            $notConnectedCalls = $queryAgentData3->notConnectedCalls ?? 0;

            $dialledPercentage = 0;
            if ($allocatedCalls > 0 && $dialledCalls > 0) {
                $dialledPercentage = round($dialledCalls / $allocatedCalls * 100);
            }

            $connectedPercentage = 0;
            if ($connectedCalls > 0 && $dialledCalls > 0) {
                $connectedPercentage = round($connectedCalls / $dialledCalls * 100);
            }


            $item->connected = new Collection();
            $item->notConnected = new Collection();

            foreach ($callReason as $row) {
                $callData = $queryResult5->where('agent_id', $agent->id)->where('name', $row->name)->first();
                $node = new stdClass;
                $node->name = $row->name;
                $node->value = $callData->count ?? 0;
                $node->properConn = null;
                if (strtolower($row->dispositions) == 'interested' || strtolower($row->dispositions) == 'customer callback') {
                    $node->properConn = 1;
                }
                if ($row->is_answered == 1) {
                    $item->connected->push($node);
                } else {
                    $item->notConnected->push($node);
                }
            }

            $properCallsPercentage = 0;
            $properConnSum = $item->connected->where('properConn', 1)->sum('value');
            if ($properConnSum > 0 && $connectedCalls > 0) {
                $properCallsPercentage = round($properConnSum / $connectedCalls * 100);
            }

            $item->branchName = $agent->branchName;
            $item->agentName = $agent->name;
            $item->allocatedCalls = $allocatedCalls;
            $item->dialledCalls = $dialledCalls;
            $item->dialledPercentage = $dialledPercentage;
            $item->connectedCalls = $connectedCalls;
            $item->notConnectedCalls = $notConnectedCalls;
            $item->connectedPercentage = $connectedPercentage;
            $item->properCallsPercentage = $properCallsPercentage;
            $result->records->add($item);
        }
        // dd($result);
        return $result;
    }

    function getReportExcel(Request $request)
    {
        $data = $this->getData($request);
        $args = $this->reportService->processrequestFilters($request);
        $params = clone $args;
        $params->type = 'sales-analysis-report-excel';
        $params->branchLabel = $this->reportService->getBranchById($args->branch);
        return $this->excelExportService->exportExcel($params, $data);
    }
}
