<?php

namespace App\Services\Reports;

use App\Models\Masters\CallSkipReason;
use App\Services\Reports\ReportService;
use App\Utilities\AppClass;
use Auth;
use Carbon\Carbon;
use Carbon\CarbonPeriod;
use DB;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;

class DailyReportService
{

    private $reportService;
    private $report;

    public function __construct(ReportService $reportService)
    {
        $this->reportService = $reportService;
        $this->report = new Collection();
        $this->header = new Collection();
    }

    public function getDailyReport(Request $request)
    {
        $args = $this->reportService->processrequestFilters($request);
        $data = new AppClass();
        $this->header = $this->getTableColumns();
        $data->rows = $this->getTableRows($args);
        $this->getreport($args, $data);
        $reports = $this->report;
        $headers = $this->header;
        $html = view('reports.daily-report-details', compact('reports', 'headers'))->render();
        $response = [
            'status' => true,
            'redirect' => null,
            'renderId' => '#report_content',
            'html' => $html,
            'tableId' => '#daily-report-datatable',
            'export' => true,
        ];
        return response()->json($response);
    }

    public function getreport($args, $data)
    {
        if ($data->rows) {
            foreach ($data->rows as $date) {
                $args->date = $date;
                $feedbacks = $this->coreDailyReport($args);
                if ($feedbacks && count($feedbacks) > 0) {
                    $this->makeReport($feedbacks, $data);
                }
            }
        }
    }

    public function makeReport($feedbacks, $data)
    {
        $reportColumns = $this->getReportColumnAttributes($data);
        $this->addDataToReport($feedbacks, $reportColumns);
    }

    public function addDataToReport($feedbacks, $reportColumns)
    {

        $tempReport = new Collection();

        foreach ($feedbacks as $feedback) {

            if (!isset($tempReport[$feedback->ref_id])) {
                $tempReport[$feedback->ref_id] = $this->getReportColumnAttributes();
            }

            $items = $tempReport->get($feedback->ref_id);

            foreach ($reportColumns as $key => $column) {

                $items->typeName = $feedback->typeName;
                $items->description = Carbon::parse($feedback->updated_at)->format('d-m-Y');

                if (is_null($feedback->skip_reason)) {
                    $items->feedbackCalls = $feedback->feedbackCount;
                    $items->complainCalls = $feedback->complaintCount;
                } else if ($feedback->skipReasonName == $key) {
                    $items->{$key} = $feedback->callCount;
                }
            }
        }

        foreach ($tempReport as $row) {
            $row->totalCalls = $this->getTotalCalls($row);
            $this->report->add($row);
        }
    }

    public function getTotalCalls($data)
    {
        $totalCalls = 0;
        foreach ($this->header as $header) {
            $totalCalls += $data->{$header->name};
        }
        $totalCalls += $data->feedbackCalls + $data->complainCalls;
        return $totalCalls;
    }

    public function getReportColumnAttributes()
    {
        $attributes = new AppClass();
        foreach ($this->header as $header) {
            $attributes->{$header->name} = null;
        }
        $attributes->typeName = null;
        $attributes->description = null;
        $attributes->feedbackCalls = null;
        $attributes->complainCalls = null;
        $attributes->totalCalls = null;
        return $attributes;
    }

    public function coreDailyReport($args)
    {
        // $args->shownBy = 'datewise';

        $sql = DB::table('beneficiary_import')
            ->join('program_master', 'program_master.id', 'beneficiary_import.program_id')
            ->join('users', 'users.id', 'beneficiary_import.updated_by')
            ->join('city_master', 'city_master.id', 'beneficiary_import.district_id')
            ->leftJoin('call_skip_reason', 'call_skip_reason.id', 'beneficiary_import.skip_reason')
            ->leftJoin('feedback', 'feedback.beneficiary_ref_id', 'beneficiary_import.id')
            ->where('beneficiary_import.status', '!=', 1)
            // ->whereDate('beneficiary_import.updated_at', '>=', $args->fromDate)
            // ->whereDate('beneficiary_import.updated_at', '<=', $args->toDate);
            ->whereDate('beneficiary_import.updated_at', $args->date);

        if ($args->program) {
            $sql->where('beneficiary_import.program_id', $args->program);
        } else if ($args->agent) {
            $sql->where('beneficiary_import.updated_by', $args->agent);
        } else if ($args->district) {
            $sql->where('beneficiary_import.district_id', $args->district);
        }

        $sql->groupBy('beneficiary_import.skip_reason');

        if (!is_null(Auth::user()->role)) {
            $sql->select(
                'beneficiary_import.skip_reason',
                'beneficiary_import.updated_at',
                'call_skip_reason.name as skipReasonName',
                DB::raw('COUNT(beneficiary_import.id) as callCount'),
                DB::raw('COUNT(CASE WHEN feedback.is_compliant = 1 AND feedback.is_show = 1 THEN feedback.id END) as complaintCount'),
                DB::raw('COUNT(CASE WHEN feedback.is_compliant = 0 AND feedback.is_show = 1 THEN feedback.id END) as feedbackCount')
            );
        } else {
            $sql->select(
                'beneficiary_import.skip_reason',
                'beneficiary_import.updated_at',
                'call_skip_reason.name as skipReasonName',
                DB::raw('COUNT(beneficiary_import.id) as callCount'),
                DB::raw('COUNT(CASE WHEN feedback.is_compliant = 1 THEN feedback.id END) as complaintCount'),
                DB::raw('COUNT(CASE WHEN feedback.is_compliant = 0 THEN feedback.id END) as feedbackCount')
            );
        }

        if ($args->shownBy == 'Agentwise') {
            $sql->groupBy('beneficiary_import.updated_by');
            $sql->addSelect('users.name as typeName', 'beneficiary_import.updated_by as ref_id');
        } else if ($args->shownBy == 'Programwise') {
            $sql->groupBy('beneficiary_import.program_id');
            $sql->addSelect('program_master.name as typeName', 'beneficiary_import.program_id as ref_id');
        } else if ($args->shownBy == 'Districtwise') {
            $sql->groupBy('beneficiary_import.district_id');
            $sql->addSelect('city_master.name as typeName', 'beneficiary_import.district_id as ref_id');
        } else if ($args->shownBy == 'datewise') {
            $sql->groupBy(DB::raw('substring(beneficiary_import.updated_at,1,10)'));
            $sql->addSelect(
                DB::raw('substring(beneficiary_import.updated_at,1,10) as typeName'),
                DB::raw('substring(beneficiary_import.updated_at,1,10) as ref_id'),
            );
        }

        $data = $sql->get();
        return $data;
    }

    public function getTableColumns()
    {
        $columns = CallSkipReason::whereStatus(CallSkipReason::STATUS_ACTIVE)->select('id', 'name')->get();
        return $columns;
    }

    public function getTableRows($args)
    {
        $startDate = Carbon::createFromFormat('Y-m-d', Carbon::parse($args->fromDate)->format('Y-m-d'));
        $endDate = Carbon::createFromFormat('Y-m-d', Carbon::parse($args->toDate)->format('Y-m-d'));

        $dateRange = CarbonPeriod::create($startDate, $endDate);
        $dates = new Collection();
        foreach ($dateRange as $date) {
            $dates->add($date->format('Y-m-d'));
        }
        return $dates;
    }
}
